/**************************************************************************
 *
 * Copyright 2012 BMW Car IT GmbH
 * Copyright (C) 2012 DENSO CORPORATION and Robert Bosch Car Multimedia Gmbh
 * Copyright (C) 2012 Bayerische Motorenwerke Aktiengesellschaft
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ****************************************************************************/
#include "ilm_common.h"
#include "ilm_tools.h"
#include "ilm_types.h"
#include "ilm_configuration.h"
#include "IpcModuleLoader.h"
#include "ObjectType.h"
#include <pthread.h>
#include <sys/time.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <semaphore.h>
#include <unistd.h>
#include <time.h>
#include <pwd.h>

/*
 *=============================================================================
 * global vars
 *=============================================================================
 */
/* automatically gets assigned argv[0] */
extern char *__progname;

/* available to all client APIs, exported in ilm_common.h */
struct IpcModule gIpcModule;

/* internal use */
static pthread_t gReceiveThread;
static pthread_t gNotificationThread;
static pthread_mutex_t gInitCouterLock;
static int gInitialized = 0;

#define LM_RECEIVER_THRD_NAME "ilm_receiver"
#define LM_NOTIFICATION_THRD_NAME "ilm_notify"
/* defined here because should never be visible throw the header file
 * of the LayerManager all of the common headers between
 * client and LayerManager are available in the sysroot*/
#define LM_INIT_SEM "/lmanager_init"
#define WAIT_TIME_FOR_LM 5

/*
 *=============================================================================
 * implementation
 *=============================================================================
 */
ilmErrorTypes ilm_init()
{
    ilmErrorTypes result = ILM_FAILED;
    t_ilm_message response = 0;
    t_ilm_message command;
    sem_t* lm_init_sem = NULL;
    struct timespec ts;
    int sem_r = 0;
    int numRetries = 400;
    int waiting_time = 5*1000;/* 5 ms*/
    struct passwd *pw = NULL;
    const char* runtime_dir;

    pthread_mutex_lock(&gInitCouterLock);
    if (gInitialized)
    {
        gInitialized++;
        fprintf(stderr,"ilm_init() was called, but initialization was skipped."
                "gInitCount is: %d\n",gInitialized);
        pthread_mutex_unlock(&gInitCouterLock);
        return ILM_SUCCESS;
    }

    runtime_dir = getenv("LM_SHARED_DIR");
    if (!runtime_dir) {
        /*implement the synchronization here */

        /* create a semaphore with value 0
         * every client will block till LayerManager post the semaphore*/
        lm_init_sem = sem_open(LM_INIT_SEM, O_CREAT, 0666, 0);
        while (lm_init_sem == SEM_FAILED || lm_init_sem == NULL)
        {
            if ((errno == EACCES) && (numRetries > 0))
            {
                printf("ilm_init "
                       "could not OPEN a lmanager sem,"
                       "retry cnt: %d\n",numRetries);
                usleep(waiting_time);
                numRetries--;
                lm_init_sem = sem_open(LM_INIT_SEM, O_CREAT, 0666, 0);
            }
            else
            {
                printf("ilm_init"
                       "could not OPEN a lmanager sem giving up,"
                       "errno: %d,  retry cnt: %d\n",errno,numRetries);
                pthread_mutex_unlock(&gInitCouterLock);
                return ILM_ERROR_ON_CONNECTION;
            }
        }
        /*change the permission, everyone can access
         *ignore the return value:
         *if user root creates a semaphore chmod will fail here*/
        chmod("/dev/shm/sem.lmanager_init", 0666);

        /*change the own to user [system] so LM has the rights to unlink the semaphore*/
        pw = getpwnam("system");
        if (pw != NULL)
            chown("/dev/shm/sem.lmanager_init",pw->pw_uid,-1);

        if (lm_init_sem == NULL)
        {
            printf("ilm_init: FAILED TO OPEN LM INIT SEMAPHORE\n");
            pthread_mutex_unlock(&gInitCouterLock);
            return result;
        }

        clock_gettime(CLOCK_REALTIME, &ts);
        ts.tv_sec += WAIT_TIME_FOR_LM;
        /* first time the semaphore is posted from LayerManager
         * wait for maximum WAIT_TIME_FOR_LM*/
        while ((sem_r = sem_timedwait(lm_init_sem, &ts)) == -1 && errno == EINTR)
            continue; /* Restart if interrupted */

        if (!sem_r)
        {
            /* if semaphore is open just post it again
             * that next client can also continue with initialization
             * (eventually there are more that one client waiting for LayerManager init)*/
            sem_post(lm_init_sem);
            sem_close(lm_init_sem);
        }
        else
        {
            printf("ilm_init: TIMEOUT IN WAITING FOR LAYERMANAGER, errno:%d\n",errno);
            sem_close(lm_init_sem);
            pthread_mutex_unlock(&gInitCouterLock);
            return result;
        }
    }

    initNotificationCallbacks();

    if (loadIpcModule(&gIpcModule))
    {
        int pid = getpid();

        if (gIpcModule.initClientMode())
        {
            pthread_attr_t notificationThreadAttributes;
            int ret;

            result = ILM_SUCCESS;

            init_msg_queue(&notificationQueue, MAX_THREAD_SYNC_QUEUE_SIZE);

            init_msg_queue(&incomingQueue, MAX_THREAD_SYNC_QUEUE_SIZE);

            if (notificationQueue.maxSize == 0)
            {
                printf("failed to allocate queue\n");
                pthread_mutex_unlock(&gInitCouterLock);
                return result;
            }

            pthread_mutex_init(&gSendReceiveLock, NULL);

            pthread_attr_init(&notificationThreadAttributes);
            pthread_attr_setdetachstate(&notificationThreadAttributes,
                                        PTHREAD_CREATE_JOINABLE);

            ret = pthread_create(&gReceiveThread,
                                    &notificationThreadAttributes,
                                    receiveThreadLoop,
                                    NULL);
            if (0 != ret)
            {
                result = ILM_FAILED;
                printf("Failed to start internal receive thread. returned %d = %s\n", ret, (ret == EAGAIN ? "EAGAIN" : "EINVAL"));
                pthread_mutex_unlock(&gInitCouterLock);
                return result;
            }

            ret = pthread_setname_np(gReceiveThread,LM_RECEIVER_THRD_NAME);
            if (0 != ret)
            {
                result = ILM_FAILED;
                printf("Failed to set name for ReceiverThread\n");
                pthread_mutex_unlock(&gInitCouterLock);
                return result;
            }

            ret = pthread_create(&gNotificationThread,
                                    &notificationThreadAttributes,
                                    notificationThreadLoop,
                                    NULL);
            if (0 != ret)
            {
                result = ILM_FAILED;
                printf("Failed to start internal notification thread. returned %d = %s\n", ret, (ret == EAGAIN ? "EAGAIN" : "EINVAL"));
                pthread_mutex_unlock(&gInitCouterLock);
                return result;
            }

            ret = pthread_setname_np(gNotificationThread, LM_NOTIFICATION_THRD_NAME);
            if (0 != ret)
            {
                result = ILM_FAILED;
                printf("Failed to set name for NotificationThread\n");
                pthread_mutex_unlock(&gInitCouterLock);
                return result;
            }

        }
        else
        {
            result = ILM_FAILED;
            printf("Failed to initialize Client Ipc Module");
            pthread_mutex_unlock(&gInitCouterLock);
            return result;
        }

        command = gIpcModule.createMessage("ServiceConnect");
        if (command
                && gIpcModule.appendUint(command, pid)
                && gIpcModule.appendString(command, __progname)
                && sendAndWaitForResponse(command, &response, RESPONSE_TIMEOUT_IN_MS, &result))
        {
            result = ILM_SUCCESS;
        }
        else
        {
            result = ILM_FAILED;
            printf("Failed to connect to LayerManagerService.");
        }
        gIpcModule.destroyMessage(response);
        gIpcModule.destroyMessage(command);
    }

    if (result == ILM_SUCCESS)
    {
        gInitialized++;
    }
    pthread_mutex_unlock(&gInitCouterLock);

    return result;
}

t_ilm_bool ilm_isInitialized()
{
    if (gInitialized >= 1)
    {
        return ILM_TRUE;
    }
    else
    {
        return ILM_FALSE;
    }
}

ilmErrorTypes ilm_commitChanges()
{
    ilmErrorTypes returnValue = ILM_FAILED;

    t_ilm_message response = 0;
    t_ilm_message command = gIpcModule.createMessage("CommitChanges");

    if (command
        && sendAndWaitForResponse(command, &response, RESPONSE_TIMEOUT_IN_MS, &returnValue))
    {
        returnValue = ILM_SUCCESS;
    }
    gIpcModule.destroyMessage(response);
    gIpcModule.destroyMessage(command);
    return returnValue;
}

ilmErrorTypes ilm_destroy()
{
    ilmErrorTypes result = ILM_FAILED;
    void* threadReturnValue = NULL;

    pthread_mutex_lock(&gInitCouterLock);
    if (gInitialized == 1)
    {
        t_ilm_message response = 0;
        t_ilm_message command = gIpcModule.createMessage("ServiceDisconnect");
        if (command
            && gIpcModule.appendUint(command, getpid())
            && sendAndWaitForResponse(command, &response, RESPONSE_TIMEOUT_IN_MS, &result))
        {
            result = ILM_SUCCESS;
        }
        gIpcModule.destroyMessage(response);
        gIpcModule.destroyMessage(command);

        /* cancel worker threads */
        pthread_cancel(gReceiveThread);
        pthread_cancel(gNotificationThread);

        pthread_join(gReceiveThread, &threadReturnValue);
        pthread_join(gNotificationThread, &threadReturnValue);

        pthread_mutex_unlock(&gSendReceiveLock);

        pthread_mutex_destroy(&gSendReceiveLock);

        gIpcModule.destroy();

        destroy_msg_queue(&notificationQueue);
        destroy_msg_queue(&incomingQueue);

        gInitialized--;
        pthread_mutex_unlock(&gInitCouterLock);

        return result;
    }
    else
    {
        if (gInitialized > 1)
        {
            gInitialized--;
            fprintf(stderr,"ilm_destroy() was called, de-initialization skipped."
                            "gInitCount is: %d\n",gInitialized);
        }
        pthread_mutex_unlock(&gInitCouterLock);
        return ILM_SUCCESS;
    }
}
/*
* called when library is loaded
*/
void __attribute__ ((constructor))global_init(void)
{
    pthread_mutex_init(&gInitCouterLock,NULL);
}


/*
* called when library is unloaded
*/
void __attribute__ ((destructor))global_destroy(void)
{
    pthread_mutex_unlock(&gInitCouterLock);
    pthread_mutex_destroy(&gInitCouterLock);
}
